<?php
/**
 * This is a PHP library that handles calling reCAPTCHA.
 *
 * @ignore
 * @copyright Copyright (c) 2015, Google Inc.
 * @link      http://www.google.com/recaptcha
 * @internal
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

namespace ReCaptcha\RequestMethod;

use ReCaptcha\RequestMethod;
use ReCaptcha\RequestParameters;

/**
 * Sends cURL request to the reCAPTCHA service.
 * Note: this requires the cURL extension to be enabled in PHP
 *
 * @see http://php.net/manual/en/book.curl.php
 *
 * @ignore
 */
class CurlPost implements RequestMethod {

	/**
	 * URL to which requests are sent via cURL.
	 *
	 * @const string
	 */
	const SITE_VERIFY_URL = 'https://www.google.com/recaptcha/api/siteverify';

	/**
	 * Curl connection to the reCAPTCHA service
	 *
	 * @var Curl
	 */
	private $curl;

	public function __construct( Curl $curl = null ) {
		if ( ! is_null( $curl ) ) {
			$this->curl = $curl;
		} else {
			$this->curl = new Curl();
		}
	}

	/**
	 * Submit the cURL request with the specified parameters.
	 *
	 * @param RequestParameters $params Request parameters
	 * @return string Body of the reCAPTCHA response
	 */
	public function submit( RequestParameters $params ) {
		$handle = $this->curl->init( self::SITE_VERIFY_URL );

		$options = array(
			CURLOPT_POST           => true,
			CURLOPT_POSTFIELDS     => $params->toQueryString(),
			CURLOPT_HTTPHEADER     => array(
				'Content-Type: application/x-www-form-urlencoded',
			),
			CURLINFO_HEADER_OUT    => false,
			CURLOPT_HEADER         => false,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_SSL_VERIFYPEER => true,
		);
		$this->curl->setoptArray( $handle, $options );

		$response = $this->curl->exec( $handle );
		$this->curl->close( $handle );

		return $response;
	}
}
